# ==============================================================================
# SCRIPT:         TransformApplierUBV.py
#
# AUTHOR:         Nikola Antonov
# AFFILIATION:    Meshtitsa Observatory / AAVSO
# CONTACT:        nikola.antonov@gmail.com
# DATE:           September 11, 2025
# VERSION:        1.0
# ==============================================================================
#
# DESCRIPTION:
# ------------------------------------------------------------------------------
# This script performs photometric transformation for U, B, and V filter data.
# It converts instrumental magnitudes from AAVSO Extended File Format output files
# into the standard Johnson-Cousins UBV photometric system.
#
# The transformation uses an iterative process to solve the equations for U, B,
# and V magnitudes, which typically depend on the (U-B) and (B-V) color
# indices. Coefficients are loaded from a 'VPhot.ini' file.
#
# The script also calculates and visualizes the U-B and B-V color indices and
# estimates the star's effective temperature from the B-V index.
#
# ==============================================================================
#
# WORKFLOW & USAGE:
# ------------------------------------------------------------------------------
# 1. PREPARE INPUT FILES:
#    - Place the script in a directory containing the following files:
#      - 'U.txt': AAVSO Extended File Format photometry data for the U-band. Must contain a NOTES
#                 column with VMAGINS, CMAGINS, and CREFMAG tags.
#      - 'B.txt': AAVSO Extended File Format photometry data for the B-band. Must contain a NOTES
#                 column with VMAGINS, CMAGINS, and CREFMAG tags.
#      - 'V.txt': AAVSO Extended File Format photometry data for the V-band. Must contain a NOTES
#                 column with VMAGINS, CMAGINS, and CREFMAG tags.
#      - 'VPhot.ini': A configuration file with transformation coefficients
#                     for the UBV system (e.g., Tu_ub, Tb_ub, Tb_bv, Tv_bv).
#
# 2. CONFIGURE SCRIPT (if needed):
#    - The input and output filenames are hardcoded in the main() function
#      call at the bottom of the script.
#
# 3. EXECUTE:
#    - Run the script from your terminal:
#      $ python TransformApplierUBV.py
#
# 4. REVIEW OUTPUT:
#    - The script will generate the following files:
#      - 'Transformed_UBV.xlsx': Excel sheet with final transformed data.
#      - '[ObjectName]_[YYYYMMDD]_UBV.txt': AAVSO-compatible text file.
#
# ==============================================================================
#
# DEPENDENCIES:
# ------------------------------------------------------------------------------
# - pandas, numpy, matplotlib, scipy
#
# ==============================================================================

import pandas as pd
import configparser
import re
from datetime import datetime

aij_columns = [
    "Name", "DATE", "MAG", "MERR", "FILT", "TRANS", "MTYPE", "CNAME",
    "CMAG", "KNAME", "KMAG", "AMASS", "GROUP", "CHART", "NOTES"
]

def parse_notes_column(df):
    def extract_mags(notes):
        kv = dict(re.findall(r"(VMAGINS|CMAGINS|CREFMAG)=([-+]?[0-9]*\.?[0-9]+)", str(notes)))
        return {
            'VarInstMag': float(kv.get("VMAGINS", "nan")),
            'CompInstMag': float(kv.get("CMAGINS", "nan")),
            'CompCatMag': float(kv.get("CREFMAG", "nan")),
        }
    return df.join(df['NOTES'].apply(extract_mags).apply(pd.Series))

def load_aij_file(filepath):
    df = pd.read_csv(filepath, names=aij_columns, comment="#")
    df = parse_notes_column(df)
    df['Err'] = pd.to_numeric(df['MERR'], errors='coerce')
    return df

def load_transform_coeffs(ini_file):
    config = configparser.ConfigParser()
    config.read(ini_file)
    return {
        'Tbv': float(config['Coefficients']['Tbv']),
        'Tv_bv': float(config['Coefficients']['Tv_bv']),
        'Tb_bv': float(config['Coefficients']['Tb_bv']),
        'Tvr': float(config['Coefficients']['Tvr']),
        'Tr_vr': float(config['Coefficients']['Tr_vr']),
        'Tv_vr': float(config['Coefficients']['Tv_vr']),
        'Tub': float(config['Coefficients']['Tub']),
        'Tu_ub': float(config['Coefficients']['Tu_ub']),
        'Tb_ub': float(config['Coefficients']['Tb_ub'])
    }

def iterative_transform_by_time(df_u, df_b, df_v, coeffs, iterations=4):
    results = []

    df_b_only = df_b[df_b['FILT'] == 'B']
    df_v_only = df_v[df_v['FILT'] == 'V']
    df_u_only = df_u[df_u['FILT'] == 'U']

    for _, v in df_v_only.iterrows():
        b_match = df_b_only.iloc[(df_b_only['DATE'] - v['DATE']).abs().argsort()[:1]].iloc[0]
        bs, bc, Bc = b_match['VarInstMag'], b_match['CompInstMag'], b_match['CompCatMag']
        vs, vc, Vc = v['VarInstMag'], v['CompInstMag'], v['CompCatMag']
        Bs, Vs = bs, vs
        for _ in range(iterations):
            Bs = Vs + (Bc - Vc) + coeffs['Tbv'] * ((bs - vs) - (bc - vc))
            Vs = vs + (Vc - vc) + coeffs['Tv_bv'] * ((Bs - Vs) - (Bc - Vc))
        row = v.copy()
        row['MAG_ORIG'] = row['MAG']
        row['MAG'] = round(Vs, 3)
        row['TRANS'] = 'YES'
        results.append(row)

    for _, b in df_b_only.iterrows():
        v_match = df_v_only.iloc[(df_v_only['DATE'] - b['DATE']).abs().argsort()[:1]].iloc[0]
        bs, bc, Bc = b['VarInstMag'], b['CompInstMag'], b['CompCatMag']
        vs, vc, Vc = v_match['VarInstMag'], v_match['CompInstMag'], v_match['CompCatMag']
        Bs, Vs = bs, vs
        for _ in range(iterations):
            Bs = Vs + (Bc - Vc) + coeffs['Tbv'] * ((bs - vs) - (bc - vc))
            Vs = vs + (Vc - vc) + coeffs['Tv_bv'] * ((Bs - Vs) - (Bc - Vc))
        row = b.copy()
        row['MAG_ORIG'] = row['MAG']
        row['MAG'] = round(Bs, 3)
        row['TRANS'] = 'YES'
        results.append(row)

    for _, u in df_u_only.iterrows():
        if df_v_only.empty or df_b_only.empty:
            continue

        v_match = df_v_only.iloc[(df_v_only['DATE'] - u['DATE']).abs().argsort()[:1]].iloc[0]
        b_match = df_b_only.iloc[(df_b_only['DATE'] - u['DATE']).abs().argsort()[:1]].iloc[0]

        us, uc, Uc = u['VarInstMag'], u['CompInstMag'], u['CompCatMag']
        vs, vc, Vc = v_match['VarInstMag'], v_match['CompInstMag'], v_match['CompCatMag']
        bs, bc, Bc = b_match['VarInstMag'], b_match['CompInstMag'], b_match['CompCatMag']

        Bs, Vs, Us = bs, vs, us

        for _ in range(iterations):
            Bs = Vs + (Bc - Vc) + coeffs['Tbv'] * ((bs - vs) - (bc - vc))
            Vs = vs + (Vc - vc) + coeffs['Tv_bv'] * ((Bs - Vs) - (Bc - Vc))
            Us = Bs + (Uc - Bc) + coeffs['Tub'] * ((us - bs) - (uc - bc))

        row = u.copy()
        row['MAG_ORIG'] = row['MAG']
        row['MAG'] = round(Us, 3)
        row['TRANS'] = 'YES'
        results.append(row)
    
    return pd.DataFrame(results).sort_values(by='DATE').reset_index(drop=True)

def save_to_excel(df, output_path):
    df.to_excel(output_path, index=False)
    print(f"✅ Saved to: {output_path}")

def save_transformed_txt(df, v_header_source, filters="UBV"):
    object_name = str(df['Name'].dropna().unique()[0]).replace(" ", "_")
    today = datetime.today().strftime("%Y%m%d")
    filename = f"{object_name}_{today}_{filters}.txt"

    with open(v_header_source, 'r') as f:
        header_lines = [line.strip() for line in f if line.startswith("#")]

    with open(filename, "w") as f:
        for line in header_lines:
            f.write(line + "\n")
        df[aij_columns].to_csv(f, index=False, header=False)

    print(f"📄 Transformed file saved as: {filename}")

def main(u_file, b_file, v_file, ini_file, output_excel):
    df_u = load_aij_file(u_file)
    df_b = load_aij_file(b_file)
    df_v = load_aij_file(v_file)
    coeffs = load_transform_coeffs(ini_file)
    df_transformed = iterative_transform_by_time(df_u, df_b, df_v, coeffs)
    save_to_excel(df_transformed, output_excel)
    save_transformed_txt(df_transformed, v_header_source=v_file)

main("U.txt", "B.txt", "V.txt", "VPhot.ini", "Transformed_UBV.xlsx")


