# ==============================================================================
# SCRIPT:         TransformApplierBV.py
#
# AUTHOR:         Nikola Antonov
# AFFILIATION:    Meshtitsa Observatory / AAVSO
# CONTACT:        nikola.antonov@gmail.com
# DATE:           September 11, 2025
# VERSION:        1.0
# ==============================================================================
#
# DESCRIPTION:
# ------------------------------------------------------------------------------
# This script performs photometric transformation for B and V filter data.
# It is designed to convert instrumental magnitudes from AAVSO Extended File Format
# output files into the standard Johnson-Cousins photometric system.
#
# The transformation is achieved through an iterative process that solves the
# transformation equations simultaneously for both filters, using coefficients
# defined in a 'VPhot.ini' file.
#
# The script also calculates and visualizes the B-V color index and estimates
# the star's effective temperature (T_eff) using the Ballesteros (2012)
# formula.
#
# ==============================================================================
#
# WORKFLOW & USAGE:
# ------------------------------------------------------------------------------
# 1. PREPARE INPUT FILES:
#    - Place the script in a directory containing the following files:
#      - 'B.txt': AAVSO Extended File Format for the B-band. Must contain a NOTES
#                 column with VMAGINS, CMAGINS, and CREFMAG tags.
#      - 'V.txt': AAVSO Extended File Format for the V-band, with the same NOTES
#                 column format.
#      - 'VPhot.ini': A configuration file with a [Coefficients] section
#                     containing Tbv, Tv_bv, and Tb_bv keys.
#
# 2. CONFIGURE SCRIPT (if needed):
#    - The input and output filenames are hardcoded in the main() function
#      call at the bottom of the script. Modify them if necessary.
#
# 3. EXECUTE:
#    - Run the script from your terminal:
#      $ python TransformApplierBV.py
#
# 4. REVIEW OUTPUT:
#    - The script will generate the following files:
#      - 'Transformed_BV.xlsx': An Excel sheet with the final transformed
#        magnitudes and original data for both filters.
#      - '[ObjectName]_[YYYYMMDD]_BV.txt': A new AAVSO-compatible text file
#        containing the transformed data, ready for submission.
#      - 'BV_Index.xlsx': An Excel sheet with the time-series data for the
#        B-V color index and estimated effective temperature.
#
#    - The script will also display the following plots:
#      - A multi-filter light curve with the transformed magnitudes.
#      - The B-V color index plotted over time with polynomial and spline fits.
#      - The estimated effective temperature plotted over time.
#
# ==============================================================================
#
# DEPENDENCIES:
# ------------------------------------------------------------------------------
# - pandas
# - numpy
# - matplotlib
# - scipy
#
# To install them, run:
# $ pip install pandas numpy matplotlib scipy
#
# ==============================================================================

import pandas as pd
import configparser
import re
import numpy as np
import matplotlib.pyplot as plt
from datetime import datetime
from scipy.interpolate import UnivariateSpline

aij_columns = [
    "Name", "DATE", "MAG", "MERR", "FILT", "TRANS", "MTYPE", "CNAME",
    "CMAG", "KNAME", "KMAG", "AMASS", "GROUP", "CHART", "NOTES"
]

def parse_notes_column(df):
    def extract_mags(notes):
        kv = dict(re.findall(r"(VMAGINS|CMAGINS|CREFMAG)=([-+]?[0-9]*\.?[0-9]+)", str(notes)))
        return {
            'VarInstMag': float(kv.get("VMAGINS", "nan")),
            'CompInstMag': float(kv.get("CMAGINS", "nan")),
            'CompCatMag': float(kv.get("CREFMAG", "nan")),
        }
    return df.join(df['NOTES'].apply(extract_mags).apply(pd.Series))

def load_aij_file(filepath):
    df = pd.read_csv(filepath, names=aij_columns, comment="#")
    df = parse_notes_column(df)
    df['Err'] = pd.to_numeric(df['MERR'], errors='coerce')
    return df

def load_transform_coeffs(ini_file):
    config = configparser.ConfigParser()
    config.read(ini_file)
    return {
        'Tbv': float(config['Coefficients']['Tbv']),
        'Tv_bv': float(config['Coefficients']['Tv_bv']),
        'Tb_bv': float(config['Coefficients']['Tb_bv']),
    }

def iterative_transform_by_time(df_b, df_v, coeffs, iterations=6):
    results = []

    for _, v in df_v.iterrows():
        b_match = df_b.iloc[(df_b['DATE'] - v['DATE']).abs().argsort()[:1]].iloc[0]
        bs, bc, Bc = b_match['VarInstMag'], b_match['CompInstMag'], b_match['CompCatMag']
        vs, vc, Vc = v['VarInstMag'], v['CompInstMag'], v['CompCatMag']
        Bs = bs
        Vs = vs
        for _ in range(iterations):
            Bs = Vs + (Bc - Vc) + coeffs['Tbv'] * ((bs - vs) - (bc - vc))
            Vs = vs + (Vc - vc) + coeffs['Tv_bv'] * ((Bs - Vs) - (Bc - Vc))
        row = v.copy()
        row['MAG_ORIG'] = row['MAG']
        row['MAG'] = round(Vs, 3)
        row['TRANS'] = 'YES'
        results.append(row)

    for _, b in df_b.iterrows():
        v_match = df_v.iloc[(df_v['DATE'] - b['DATE']).abs().argsort()[:1]].iloc[0]
        bs, bc, Bc = b['VarInstMag'], b['CompInstMag'], b['CompCatMag']
        vs, vc, Vc = v_match['VarInstMag'], v_match['CompInstMag'], v_match['CompCatMag']
        Bs = bs
        Vs = vs
        for _ in range(iterations):
            Bs = Vs + (Bc - Vc) + coeffs['Tbv'] * ((bs - vs) - (bc - vc))
            Vs = vs + (Vc - vc) + coeffs['Tv_bv'] * ((Bs - Vs) - (Bc - Vc))
        row = b.copy()
        row['MAG_ORIG'] = row['MAG']
        row['MAG'] = round(Bs, 3)
        row['TRANS'] = 'YES'
        results.append(row)

    return pd.DataFrame(results).sort_values(by='DATE').reset_index(drop=True)

def estimate_temperature_from_bv(bv):
    if np.isnan(bv):
        return np.nan

    # Balesteros formula (Balesteros, 2012, New insights on black bodies)
    return 4600 * ((1 / (0.92 * bv + 1.7)) + (1 / (0.92 * bv + 0.62)))

def calculate_bv_index(df, max_time_diff=0.01):
    df_b = df[df['FILT'] == 'B'].copy()
    df_v = df[df['FILT'] == 'V'].copy()

    bv_data = []

    for _, v_row in df_v.iterrows():
        b_candidates = df_b[(df_b['DATE'] - v_row['DATE']).abs() <= max_time_diff]
        if not b_candidates.empty:
            b_row = b_candidates.iloc[(b_candidates['DATE'] - v_row['DATE']).abs().argsort().iloc[0]]
            bv_index = b_row['MAG'] - v_row['MAG']
            b_err = b_row.get('Err', np.nan)
            v_err = v_row.get('Err', np.nan)
            temperature = estimate_temperature_from_bv(bv_index)
            bv_data.append({
                'DATE': v_row['DATE'],
                'B_MAG': b_row['MAG'],
                'V_MAG': v_row['MAG'],
                'B_ERR': b_err,
                'V_ERR': v_err,
                'B_V': round(bv_index, 3),
                'T_eff': round(temperature, 1)
            })

    return pd.DataFrame(bv_data)

def save_to_excel(df, output_path):
    df.to_excel(output_path, index=False)
    print(f"✅ Saved to: {output_path}")

def save_transformed_txt(df, v_header_source, filters="BV"):
    object_name = str(df['Name'].dropna().unique()[0]).replace(" ", "_")
    today = datetime.today().strftime("%Y%m%d")
    filename = f"{object_name}_{today}_{filters}.txt"

    with open(v_header_source, 'r') as f:
        header_lines = [line.strip() for line in f if line.startswith("#")]

    with open(filename, "w") as f:
        for line in header_lines:
            f.write(line + "\n")
        df[aij_columns].to_csv(f, index=False, header=False)

    print(f"📄 Transformed file saved as: {filename}")

def plot_light_curve(df):
    plt.figure(figsize=(10, 6))
    plt.gca().invert_yaxis()

    filter_colors = {
        'U': 'violet', 'B': 'blue', 'V': 'green', 'R': 'red'
    }

    for filt, group in df.groupby('FILT'):
        color = filter_colors.get(filt.upper(), 'black')
        plt.errorbar(
            group['DATE'], group['MAG'], yerr=group['Err'],
            fmt='o', ecolor=color, color=color, label=filt,
            capsize=2, markersize=4
        )

    plt.xlabel('JD (DATE)')
    plt.ylabel('Magnitude')
    plt.title('EV Lac (UT2025-07-15)')
    plt.grid(True)
    plt.legend(title='Filter')
    plt.tight_layout()
    plt.figtext(0.99, 0.01, f"N. Antonov, AO Meshtitsa; 0.25-m F/4.8 Newtonian; ASI533MM Pro", ha='right', va='center', fontsize=10, style='italic')
    plt.show()

def plot_bv_index(df_bv, poly_degree=4):
    plt.figure(figsize=(10, 6))
    x = df_bv['DATE'].values
    y = df_bv['B_V'].values
    x_mean = x.mean()
    x_centered = x - x_mean

    coeffs = np.polyfit(x_centered, y, deg=poly_degree)
    poly_func = np.poly1d(coeffs)
    x_fit = np.linspace(x.min(), x.max(), 500)
    y_fit = poly_func(x_fit - x_mean)

    plt.plot(x, y, marker='o', linestyle='-', color='orange', label='Observed B−V')
    plt.plot(x_fit, y_fit, color='black', linestyle='--', linewidth=2, label=f'{poly_degree}° Polynomial Fit')

    plt.xlabel('JD (DATE)')
    plt.ylabel('B − V')
    plt.title('B−V Color Index Over Time with Polynomial Fit')
    plt.grid(True)
    plt.legend()
    plt.tight_layout()
    plt.show()

def plot_bv_index_spline(df_bv, spline_smoothing=0.02):
    plt.figure(figsize=(10, 6))
    x = df_bv['DATE'].values
    y = df_bv['B_V'].values

    if 'B_ERR' in df_bv.columns and 'V_ERR' in df_bv.columns:
        err = np.sqrt(df_bv['B_ERR']**2 + df_bv['V_ERR']**2)
        weights = 1 / err
    else:
        weights = None

    spline = UnivariateSpline(x, y, w=weights, s=spline_smoothing)
    x_smooth = np.linspace(x.min(), x.max(), 500)
    y_smooth = spline(x_smooth)

    plt.plot(x, y, marker='o', linestyle='-', color='orange', label='Observed B−V')
    plt.plot(x_smooth, y_smooth, color='blue', linestyle='-', linewidth=2, label='Spline Fit (weighted)')

    plt.xlabel('JD (DATE)')
    plt.ylabel('B − V')
    plt.title('B−V Color Index with Weighted Spline Fit')
    plt.grid(True)
    plt.legend()
    plt.tight_layout()
    plt.show()

def plot_temperature(df_bv):
    plt.figure(figsize=(10, 6))
    plt.plot(df_bv['DATE'], df_bv['T_eff'], marker='o', linestyle='-', color='red')
    plt.xlabel('JD (DATE)')
    plt.ylabel('Estimated $T_{eff}$ [K]')
    plt.title('Estimated Effective Temperature Over Time')
    plt.grid(True)
    plt.tight_layout()
    plt.show()

def main(b_file, v_file, ini_file, output_excel):
    df_b = load_aij_file(b_file)
    df_v = load_aij_file(v_file)
    coeffs = load_transform_coeffs(ini_file)
    df_transformed = iterative_transform_by_time(df_b, df_v, coeffs)
    save_to_excel(df_transformed, output_excel)
    save_transformed_txt(df_transformed, v_header_source=v_file)
    plot_light_curve(df_transformed)

    df_bv_index = calculate_bv_index(df_transformed)
    df_bv_index.to_excel("BV_Index.xlsx", index=False)
    plot_bv_index(df_bv_index)
    plot_bv_index_spline(df_bv_index)
    plot_temperature(df_bv_index)

main("B.txt", "V.txt", "VPhot.ini", "Transformed_BV.xlsx")
